model GasForce3 "Rough approximation of gas force in a cylinder"
  import SI = Modelica.SIunits;
  extends Modelica.Mechanics.Translational.Interfaces.PartialCompliant;
  parameter SI.Length L = 0.5 "Length of cylinder";
  parameter SI.Length d = 0.1 "diameter of cylinder";
  parameter SI.Volume k0 = 0.0 "Volume V = k0 + k1*(1-x), with x = 1 - s_rel/L";
  parameter SI.Volume k1 = 1 "Volume V = k0 + k1*(1-x), with x = 1 - s_rel/L";
  constant Real pi = Modelica.Constants.pi;
  Real x "Normalized position of cylinder (= 1 - s_rel/L)";
  Modelica.SIunits.AbsolutePressure press "Cylinder pressure";
  SI.Volume V;
  SI.Temperature T;
  SI.Velocity v_rel;
protected
  Real R = Modelica.Constants.R;
  constant Real L0 = 0.5;
  Real n;
equation
  x = 1 - s_rel / L0;
  v_rel = der(s_rel);
  press = 100000.0 * (if v_rel < 0 then if x < 0.987 then 177.4132 * x ^ 4 - 287.2189 * x ^ 3 + 151.8252 * x ^ 2 - 24.9973 * x + 2.4 else 2836360 * x ^ 4 - 10569296 * x ^ 3 + 14761814 * x ^ 2 - 9158505 * x + 2129670 else if x > 0.93 then -3929704 * x ^ 4 + 14748765 * x ^ 3 - 20747000 * x ^ 2 + 12964477 * x - 3036495 else 145.93 * x ^ 4 - 131.707 * x ^ 3 + 17.3438 * x ^ 2 + 17.9272 * x + 2.4);
  f = -(press * pi * d ^ 2) / 4;
  V = ((k0 + k1 * (1 - x)) * pi * d ^ 2) / 4;
  n = (L * pi * (0.5 * d) ^ 2 * 1000) / 22.4;
  press * V = n * R * T;
  assert(s_rel >= -1e-012, "flange_b.s - flange_a.s (= " + String(s_rel) + ") >= 0 required for GasForce component.
" + "Most likely, the component has to be flipped.");
  assert(s_rel <= L + 1e-012, " flange_b.s - flange_a.s (= " + String(s_rel) + " <= L (" + String(L) + ") required for GasForce component.
" + "Most likely, parameter L is not correct.");
  annotation(Icon(coordinateSystem(preserveAspectRatio = true, extent = {{-100,-100},{100,100}}, grid = {1,1}), graphics = {Rectangle(extent = {{-90,50},{90,-50}}, lineColor = {0,0,0}, fillColor = {192,192,192}, fillPattern = FillPattern.Solid),Polygon(points = {{-13,4},{-16,4},{-65,4},{-65,15},{-90,0},{-65,-15},{-65,-4},{-13,-4},{-13,4}}, lineColor = {255,0,0}, fillColor = {255,0,0}, fillPattern = FillPattern.Solid),Text(extent = {{-100,120},{100,60}}, textString = "%name", lineColor = {0,0,255}),Text(extent = {{-135,44},{-99,19}}, lineColor = {128,128,128}, textString = "a"),Text(extent = {{97,40},{133,15}}, lineColor = {128,128,128}, textString = "b"),Polygon(points = {{12,4},{70,4},{65,4},{65,15},{90,0},{65,-15},{65,-4},{12,-4},{12,4}}, lineColor = {255,0,0}, fillColor = {255,0,0}, fillPattern = FillPattern.Solid)}), Diagram(coordinateSystem(preserveAspectRatio = true, extent = {{-100,-100},{100,100}}, grid = {1,1}), graphics = {Rectangle(extent = {{-90,50},{90,-50}}, lineColor = {0,0,0}, fillColor = {192,192,192}, fillPattern = FillPattern.Solid),Text(extent = {{-100,120},{100,60}}, textString = "%name", lineColor = {0,0,255}),Polygon(points = {{12,5},{70,5},{65,5},{65,16},{90,1},{65,-14},{65,-3},{12,-3},{12,5}}, lineColor = {255,0,0}, fillColor = {255,0,0}, fillPattern = FillPattern.Solid),Polygon(points = {{-13,5},{-16,5},{-65,5},{-65,16},{-90,1},{-65,-14},{-65,-3},{-13,-3},{-13,5}}, lineColor = {255,0,0}, fillColor = {255,0,0}, fillPattern = FillPattern.Solid)}), Documentation(info = "<html>
<p>
The gas force in a cylinder is computed as function of the relative
distance of the two flanges. It is required that s_rel = flange_b.s - flange_a.s
is in the range
</p>
<pre>
    0 &le; s_rel &le; L
</pre>
<p>
where the parameter L is the length
of the cylinder. If this assumption is not fulfilled, an error occurs.
</p>
</html>"));
end GasForce3;

